! **********************************************************************

       subroutine rdenv(isshel)

! ----------------------------------------------------------------------
!
!     WEATHER AND CONDUCTION TRANSFER FUNCTIONS:
!        Read the weather data file and the file of conduction transfer
!        functions of building constructs.
!        Interpolate hourly weather data.
!        Calculate sines of solar altitude and azimuth angles.
!
!     December 6, 1984  Cheol Park
!     Last Updated: August 1, 1985  D. Clark
!     Updated:  October 18, 2002  Cheol Park & Mike Galler, NIST
!     Updated:  May 10, 2007  C. Park
!               Converted to Fortran 90 and some changes were made.
!
!     inputs from file  ifile7 (ctfdata.dat):
!       idstr:    ID number of structures, [1,10] (-)
!       nctfa:    Number of CTF terms, [1,20] (-)
!       norda:    Number of order, [1,5] (-)
!       u:        Overall conductance (W/m**2-K)
!       x:        Conduction transfer function (W/m**2-K)
!       y:        Conduction transfer function (W/m**2-K)
!       z:        Conduction transfer function (W/m**2-K)
!       rj:       Flux referring CTF           (W/m**2)
!
!     input from file ifile8 (weather.dat):
!       month:    Month of year (-)
!       alat:     Latitude (degree)
!       along:    Longitude (degree)
!       tzn:      Time zone number (-)
!                 =4 --- Atlantic
!                 =5 --- Eastern
!                 =6 --- Central
!                 =7 --- Mountain
!                 =8 --- Pacific
!       isflag:   Flag identifying source of solar radiation data in
!                 weather file.
!
!       hr:       Time of day (h)
!       t:        Outside air dry-bulb temperature (C)      -- toa
!       w:        Outside air humidity ratio (-)            -- woa
!       p:        Outside air barometric pressure (kPa)     -- poa
!       v:        Outside wind speed (m/s)                  -- vwind
!       soldir:   Direct normal solar radiation (W/m**2)    -- hdn
!       solsky:   Sky diffusive radiation (W/m**2)          -- hsky
!       solhoz:   Total horizontal radiation (W/m**2)       -- hhor
!
!     subprograms called:
!       SPLINE, SPEVAL
!     REFERENCES:
!
!
! **********************************************************************

      use modsim_head
      implicit none

      integer,parameter              :: nwdata=25
      logical                        :: initial_time=.true.,eof=.false.
      character(len=3),dimension(12) :: mname= &
                                   (/'Jan','Feb','Mar','Apr','May','Jun',&
                                     'Jul','Aug','Sep','Oct','Nov','Dec'/)
      character(len=28),dimension(3) :: source= &
                                       (/'weather tape                ',&
                                         'clear sky design day method ',&
                                         'cloudy sky design day method'/)

      integer                        :: ios,isshel,i,idstr,n,isflag,iday1,&
                                        nday,iday,nw,ii,month,day

      real                           :: dtr=0.0174533,tzilch,hour,tzn,&
                                        along,alat,b,decl,hss,hsr,h1,h2,&
                                        h,eqtm,cosh,coshs,cosw,cosz,sinl,&
                                        cosl,sind,cosd
      real                           :: hour1                                       
      real,dimension(nwdata)         :: hr,t,w,p,v,soldir,solsky,solhoz,&
                                        fdpt,fdpw,fdpp,fdpv,fdpd,fdps,fdph
      integer,dimension(12)          :: mdays= &
                             (/0,31,59,90,120,151,181,212,243,273,304,334/)

!      real                           :: time1,interval = 3600.0
!      namelist /nam1/time1,h1,h2,h,cosh,cosz,cosw,ssalt,ssazm

!     At initial time, open the weather, and ctf files.
!     read ctf data.

      if(initial_time) then
        initial_time=.false.
        tzero=tzilch
        hour=time/3600.+tzero
        do i=1,maxstr
          read(ifile7,*,iostat=ios) idstr,nctfa(i),norda(i),u(i)
          if(ios<0) then            ! end of file
            exit
          elseif(ios>0) then        ! error in reading file
            stop 'Error: reading ctf file.'
          endif

          read(ifile7,*) (xc(n,i),n=0,nctfa(i))
          read(ifile7,*) (yc(n,i),n=0,nctfa(i))
          read(ifile7,*) (zc(n,i),n=0,nctfa(i))
          read(ifile7,*) (rj(n,i),n=1,norda(i))
        enddo

!     Write weather-related information to simulation summary file

20      write(ifile3,1000) isshel,tshell
        1000  format(' Building shell model in superblock ',i2,':'/&
                     '   constant time step    tshell = ',f8.2/)
        read(ifile8,*) month,iday1,alat,along,tzn,isflag
        write(ifile3,2000) alat,along,iday1,mname(month),source(isflag)
        2000  format(' weather data:  latitude =',f8.3,&
                     '  longitude =',f8.3/' starting date: ',i4,1x,a3,&
                     '.'/' source:  ',a28/)

        sinl=sin(alat*dtr)
        cosl=cos(alat*dtr)
        nday=mdays(month)+iday1
        b=dtr*360.*(nday-81)/364.
        eqtm=(9.87*sin(2*b)-7.53*cos(b)-1.5*sin(b))/60.
        decl=dtr*23.45*sin(dtr*360.*(284+nday)/365.)
        sind=sin(decl)
        cosd=cos(decl)

!     Read  first set of weather data.

        iday = 0
        i = 1
        read(ifile8,*,iostat=ios) month,day,hour1,t(i),w(i),p(i),v(i),&
                                  soldir(i),solsky(i),solhoz(i)
        do i=1,nwdata
          ii=i
          read(ifile8,*,iostat=ios) month,day,hour1,t(i),w(i),p(i),v(i),&
                                  soldir(i),solsky(i),solhoz(i)
          if(ios<0) then
            if(ii<=4) then
              print *,' Error:  insufficient weather data '
              print *,' at least four records are required'
              stop ' **** simulation terminated  ********'
            endif
          elseif(ios>0) then
            stop 'Error: reading weather data file.'
          endif

!          print *, 'month,day,hour1,t(i)',month,day,hour1,t(i)

          hr(i) = real(i)
        enddo
        print *,'--  first weather data set has been read '

        nw=nwdata
        call spline(nw,hr,t,fdpt)
        call spline(nw,hr,w,fdpw)           
        call spline(nw,hr,p,fdpp)           
        call spline(nw,hr,v,fdpv)           
        call spline(nw,hr,soldir,fdpd)      
        call spline(nw,hr,solsky,fdps)      
        call spline(nw,hr,solhoz,fdph)      
                                            
      endif

      hour=time/3600.
      if(hour >= 24.0*iday) then
        hour = hour - 24.0*iday
      endif

!     Compute solar declination, equation of time, and sunset hour
!     angle, once a day.

      if(hour+tzilch>=24.) then
        tzilch=tzilch-24.
        nday=nday+1
        b=dtr*360.*(nday-81)/364.
        eqtm=(9.87*sin(2*b)-7.53*cos(b)-1.5*sin(b))/60.
        decl=dtr*23.45*sin(dtr*360.*(284+nday)/365.)
        sind=sin(decl)
        cosd=cos(decl)
        coshs=-sinl*sind/(cosl*cosd)
        hss=acos(coshs)
        hsr=-hss
      endif

      hour=hour+tzero

      if(eof .or. (hour < hr(nwdata-1))) then
         goto 70
      endif   

!     Read another set of weather data if necessary

      if(time >= iday*24.0*3600.0-tstep) then
         iday = iday +1
         backspace ifile8
         do i=1,nwdata
           read(ifile8,*,iostat=ios) month,day,hour1,t(i),w(i),p(i),v(i),&
                                  soldir(i),solsky(i),solhoz(i)                 
           if(ios>0) then
             stop 'Error: reading weather data file.'             
           endif                                                  
!           print *, 'month,day,hour1,t(i)',month,day,hour1,t(i)
           hr(i) = real(i)
         enddo

         nw=nwdata
         call spline(nw,hr,t,fdpt)
         call spline(nw,hr,w,fdpw)                
         call spline(nw,hr,p,fdpp)                
         call spline(nw,hr,v,fdpv)                
         call spline(nw,hr,soldir,fdpd)           
         call spline(nw,hr,solsky,fdps)           
         call spline(nw,hr,solhoz,fdph)           
                                                  
      endif

!     Interpolate the weather data.

70    continue
      call speval(nw,hr,t,fdpt,hour,toa)
      call speval(nw,hr,w,fdpw,hour,woa)
      call speval(nw,hr,p,fdpp,hour,poa)
      call speval(nw,hr,v,fdpv,hour,vwind)
      call speval(nw,hr,soldir,fdpd,hour,hdn)
      call speval(nw,hr,solsky,fdps,hour,hsky)
      call speval(nw,hr,solhoz,fdph,hour,hhor)

      if(woa<0.)   woa   =0.
      if(poa<0.)   poa   =0.
      if(vwind<0.) vwind =0.
      if(hdn<0.)   hdn   =0.
      if(hsky<0.)  hsky  =0.
      if(hhor<0.)  hhor  =0.

!     Solar geometry is calculated half a time step ago, to represent
!     the average solar position over the time step.  If sunrise or
!     sunset occurs during the time step, exclude the night-time
!     portion of the time step.

      if(hour>=24.) hour=hour-24.
      h1=(15.*(hour-12.+eqtm+tzn)-along)*dtr
      h2=h1-dtr*tstep/240.
      if(h2<hsr .and. h1>hsr) h2=hsr
      if(h1>hss .and. h2<hss) h1=hss
      h=0.5*(h1+h2)

!     Calculate the directional cosines

      cosh=cos(h)
      cosz=sind*sinl+cosd*cosl*cosh
      cosw=cosd*sin(h)

!     Determine the sines of solar altitude and azimuth angles.

      ssalt=cosz
      ssazm=cosw/sqrt(1.-ssalt*ssalt)
!      if(time > 460000.0 .and. mod(time, interval) < tstep) then
!      print *, 'time,time1,ssazm:', time, time1, ssazm
!      print nam1
!      endif   

      if(abs(ssazm) > 1.0) then
        ssazm = 1.0
      endif

!     Make sure the sun is turned off at night.

      if(cosz<0.) then
        hdn=0.
        hsky=0.
        hhor=0.
      endif

!     Copy interpolated weather data into state vector

      if(ntoa>0 .and. ndent(3)>=0) then
        n=ndent(3)+ntoa
        state(n)=toa
        tstate(n)=toa
      endif

      if(nwoa>0 .and. ndent(8)>=0) then
        n=ndent(8)+nwoa
        state(n)=woa
        tstate(n)=woa
      endif

      if(npoa>0 .and. ndent(1)>=0) then
        n=ndent(1)+npoa
        state(n)=poa
        tstate(n)=poa
      endif

      if(ndent(7)>=0) then
        if(ndn>0) then
          n=ndent(7)+ndn
          state(n)=hdn
          tstate(n)=hdn
        endif
        if(nsky>0) then
          n=ndent(7)+nsky
          state(n)=hsky
          tstate(n)=hsky
        endif
        if(nhor>0) then
          n=ndent(7)+nhor
          state(n)=hhor
          tstate(n)=hhor
        endif
      endif

      return
      end subroutine rdenv

! **********************************************************************

      subroutine spline(n,x,y,fdp)

! ----------------------------------------------------------------------
!
!     SPLINE : computes the second derivatives needed in cubic
!             spline interpolation. The original program was written by
!              J. H. Ferziger Ref.[1]. A little modification is made.
!
!     July 18, 1984 C.P.
!     Update to Fortran 90 May 10, 2007 C. Park
!
!     N:         Number of data points
!     X:         Array containing the values of the independent variable
!                (Assume to be in ascending order)
!     Y:         Array containing the values of the function at the data
!                points given in the X array
!
!     FDP:       Output array which contains the second derivatives of
!                the interpolating cubic spline.
!
!     REFERENCE:
!       [1]  Joel H. Ferziger, "Numerical Methods for Engineering
!                    Application," John Wiley & Sons, 1981, pp.17-18.
!
! **********************************************************************

      implicit none

      integer                          :: i,n
      integer,parameter                :: nmax=25
      real                             :: lamda,t
      real,dimension(nmax),intent(in)  :: x,y
      real,dimension(nmax),intent(out) :: fdp
      real,dimension(nmax)             :: a,b,c,r

!     Compute the coefficients and the rhs of the equations.
!     this routine uses the cantilever condition. the parameter
!     lamda is set to 1. but this can be user-modified.

!     a,b,c are the three diagonals of the tridiagonal system,
!     and r is the right hand side.

      lamda = 1.
      c(1)=x(2)-x(1)
      do i=2,n-1
        c(i)=x(i+1)-x(i)
        a(i)=c(i-1)
        b(i)=2.*(a(i)+c(i))
        r(i)=6.*((y(i+1)-y(i))/c(i)-(y(i)-y(i-1))/c(i-1))
      enddo
      b(2)=b(2)+lamda*c(1)
      b(n-1)=b(n-1)+lamda*c(n-1)

!     Tridiagonal solver subroutine.
!     But the notaion is clumsy so we will solve directly.

      do i=3,n-1
        t=a(i)/b(i-1)
        b(i)=b(i)-t*c(i-1)
        r(i)=r(i)-t*r(i-1)
      enddo
      fdp(n-1)=r(n-1)/b(n-1)
      do i=2,n-2
        fdp(n-i)=(r(n-i)-c(n-i)*fdp(n-i+1))/b(n-i)
      enddo
      fdp(1)=lamda*fdp(2)
      fdp(n)=lamda*fdp(n-1)

      return
      end
! **********************************************************************

      subroutine speval(n,x,y,fdp,xx,f)

! ----------------------------------------------------------------------
!
!     SPEVAL : evaluates the cubic spline for given the derivatives
!              computed by subroutine SPLINE.
!
!     XX:        Value of independent variable for which an interpolated
!                value is reqested
!     F:         The interpolated result
!
! **********************************************************************
      implicit none

      integer                          :: i,n
      integer,parameter                :: nmax=25
      real,intent(in out)              :: xx
      real,intent(out)                 :: f
      real                             :: dxm,dxp,del
      real,dimension(nmax),intent(in)  :: x,y,fdp

!     Find the proper interval.

      do i=1,n-1
        if(xx<=x(i+1)) exit
      enddo

!     Evaluate the cubic.

      dxm=xx-x(i)
      print *, 'i == ', i
      if(i >= 25) then
        print *, ' i too high'
        endif
      dxp=x(i+1)-xx
      del=x(i+1)-x(i)
      f=fdp(i)*dxp*(dxp*dxp/del-del)/6.0&
        +fdp(i+1)*dxm*(dxm*dxm/del-del)/6.0&
        +y(i)*dxp/del+y(i+1)*dxm/del

      return
      end

